# Deploying to Live Domain - Step by Step Guide

## Prerequisites
- SSH access to your live server (cPanel hosting usually provides this)
- FTP or File Manager access
- Database credentials for live server
- Git installed on server (optional but recommended)

---

## Step 1: Prepare Local Environment

### 1.1 Backup Current Local Database
```bash
# Export local database
mysqldump -u root -p multivendor > backup_local.sql
```

### 1.2 Create Migration & Seed Files
```bash
# On your local machine, ensure all migrations are in:
# database/migrations/

# Ensure seeds are set up:
# database/seeds/ or database/seeders/
```

### 1.3 Clean Up Local Files
```bash
# Remove unnecessary files
rm -rf node_modules/
rm -rf storage/logs/*
rm -rf bootstrap/cache/*
```

---

## Step 2: Create Live Database

### Option A: Using cPanel
1. Log into cPanel
2. Go to **MySQL Databases**
3. Create new database:
   - Name: `live_db_name` (no spaces)
4. Create new MySQL user:
   - Username: `live_user`
   - Password: `strong_password_here`
5. Add user to database with **ALL PRIVILEGES**

### Option B: Via SSH
```bash
# SSH into your server first
mysql -u root -p

# In MySQL prompt:
CREATE DATABASE live_db_name;
CREATE USER 'live_user'@'localhost' IDENTIFIED BY 'strong_password_here';
GRANT ALL PRIVILEGES ON live_db_name.* TO 'live_user'@'localhost';
FLUSH PRIVILEGES;
EXIT;
```

---

## Step 3: Upload Codebase

### Option A: Via Git (Recommended)
```bash
# On live server via SSH
cd /home/username/public_html/

# Clone your repo (or create one locally first)
git clone https://github.com/yourusername/your-repo.git .

# Or if updating:
git pull origin main
```

### Option B: Via FTP/File Manager
1. Open cPanel **File Manager**
2. Navigate to `/public_html/`
3. Upload all files EXCEPT:
   - `.git/` (if exists)
   - `node_modules/`
   - `storage/logs/` (let server create)
   - `bootstrap/cache/` (let server create)

### Option C: Via SSH & Tar
```bash
# On local machine, create archive:
tar --exclude='node_modules' --exclude='.git' --exclude='storage/logs' \
    --exclude='bootstrap/cache' -czf crowdfunding.tar.gz .

# Upload to server, then extract:
scp crowdfunding.tar.gz username@yourdomain.com:/home/username/public_html/
ssh username@yourdomain.com
cd /home/username/public_html/
tar -xzf crowdfunding.tar.gz
```

---

## Step 4: Configure Environment

### 4.1 Create .env File
```bash
# SSH into server
cd /home/username/public_html/

# Copy example file
cp .env.example .env

# Edit with nano or vim
nano .env
```

### 4.2 Update .env Values
```env
APP_NAME="E-Africa Mall"
APP_ENV=production
APP_DEBUG=false  # IMPORTANT: false for live
APP_KEY=base64:YOUR_EXISTING_KEY  # Copy from local .env
APP_URL=https://yourdomain.com

# Database
DB_CONNECTION=mysql
DB_HOST=localhost
DB_PORT=3306
DB_DATABASE=live_db_name          # From Step 2
DB_USERNAME=live_user             # From Step 2
DB_PASSWORD=strong_password_here   # From Step 2

# Cache & Session
CACHE_DRIVER=file
SESSION_DRIVER=file
QUEUE_CONNECTION=sync

# Mail (optional, for notifications)
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=587
MAIL_USERNAME=your_username
MAIL_PASSWORD=your_password

# Storage (if using cloud)
# AWS_ACCESS_KEY_ID=
# AWS_SECRET_ACCESS_KEY=
# AWS_DEFAULT_REGION=us-east-1
# AWS_BUCKET=
```

### 4.3 Set File Permissions
```bash
# Make storage writable
chmod -R 775 storage/
chmod -R 775 bootstrap/cache/

# Secure .env file
chmod 600 .env
```

---

## Step 5: Install Dependencies & Migrate

### 5.1 Install Composer Dependencies
```bash
# Via SSH
cd /home/username/public_html/
composer install --no-dev --optimize-autoloader
```

### 5.2 Generate Application Key (if not set)
```bash
php artisan key:generate
```

### 5.3 Run Database Migrations
```bash
# Run all migrations
php artisan migrate --force

# Expected output:
# Migration table created successfully.
# Migrating: 2025_01_01_000000_create_users_table
# ... (more migrations)
# Migrated: (count) migrations
```

### 5.4 Seed Database (Optional)
```bash
# If you have seeders
php artisan db:seed --force

# Or specific seeder
php artisan db:seed --class=DatabaseSeeder --force
```

---

## Step 6: Verify Database Tables

### Check Business Settings Table
```bash
# SSH in and use tinker
php artisan tinker

# In tinker prompt:
> DB::table('business_settings')->count()
# Should return: integer > 0

> DB::table('business_settings')->first()
# Should show settings data

> exit
```

---

## Step 7: Clear Cache & Optimize

```bash
# Clear all caches
php artisan cache:clear
php artisan config:cache
php artisan route:cache
php artisan view:clear

# Optimize autoloading
php artisan optimize
```

---

## Step 8: Configure Web Server

### For Apache (cPanel)
1. Document Root should be: `/home/username/public_html/public`
2. `.htaccess` should be in `/public_html/public/` (usually included)
3. Make sure mod_rewrite is enabled

### For Nginx
```nginx
server {
    listen 80;
    server_name yourdomain.com;
    root /home/username/public_html/public;
    
    index index.php index.html;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php-fpm.sock;
        fastcgi_param SCRIPT_FILENAME $realpath_root$fastcgi_script_name;
        include fastcgi_params;
    }
}
```

---

## Step 9: Test Live Site

### 9.1 Basic Connectivity Test
```bash
# From local machine
curl https://yourdomain.com
# Should return HTML, not 500 error
```

### 9.2 Visit Website
1. Open browser: `https://yourdomain.com`
2. Should show homepage without errors
3. Check for images loading
4. Verify "Delivery" terminology (from earlier changes)

### 9.3 Check Error Logs
```bash
# SSH into server
tail -50 /home/username/public_html/storage/logs/laravel.log

# Should show normal GET requests, no errors
# If errors, read carefully for database/config issues
```

### 9.4 Test Key Features
- [ ] Homepage loads without 500 error
- [ ] Can browse products
- [ ] Can add to cart
- [ ] Checkout page works
- [ ] Can view orders (if logged in)
- [ ] Admin panel accessible (if needed)

---

## Step 10: SSL Certificate Setup

### If Using cPanel AutoSSL
1. Go to cPanel **AutoSSL**
2. Check if certificate is installed
3. If not, install it manually through cPanel

### If Using Let's Encrypt (Recommended)
```bash
# Via SSH with certbot
sudo certbot certonly --webroot -w /home/username/public_html/public \
    -d yourdomain.com -d www.yourdomain.com

# Auto-renew
sudo certbot renew --quiet
```

---

## Subdomain Alternative

### To Deploy to Subdomain Instead

#### 1. Create Subdomain in cPanel
- Name: `dev.yourdomain.com`
- Document Root: `public_html/dev/public`

#### 2. Upload Files to `/public_html/dev/`

#### 3. Follow Same Steps Above

#### 4. Test at: `https://dev.yourdomain.com`

**Advantages**:
- Doesn't affect main website
- Easy to delete and retry
- Perfect for testing

---

## Troubleshooting

### Problem: 500 Internal Server Error
**Solution**:
```bash
# Check logs
tail -50 storage/logs/laravel.log

# Most likely: Database connection issue
# Verify .env has correct DB credentials
# Test connection:
php artisan tinker
> DB::connection()->getPdo();
```

### Problem: Page shows blank/white
**Solution**:
```bash
# Enable debug temporarily
# Edit .env:
APP_DEBUG=true

# Refresh page, check error message
# Don't leave this on!
```

### Problem: 404 on all pages
**Solution**:
- Check document root is set to `/public`
- Verify `.htaccess` exists in `/public/`
- Check Apache `mod_rewrite` is enabled

### Problem: Images not loading
**Solution**:
- Check storage symlink: `php artisan storage:link`
- Verify file permissions on storage
- Check image paths in database

### Problem: Database migration fails
**Solution**:
```bash
# Reset and retry
php artisan migrate:reset --force
php artisan migrate --force
php artisan db:seed --force
```

---

## Maintenance After Deployment

### Regular Tasks
```bash
# Weekly
php artisan cache:clear
php artisan view:clear

# Monthly
php artisan optimize:clear

# Check logs for errors
tail -100 storage/logs/laravel.log
```

### Backup Strategy
```bash
# Daily backup (add to cron job)
mysqldump -u live_user -p'password' live_db_name > backup_$(date +%Y%m%d).sql

# Backup files monthly
tar -czf backup_files_$(date +%Y%m%d).tar.gz public_html/
```

### Update Codebase
```bash
# If using Git
git pull origin main
composer install --no-dev
php artisan migrate --force
php artisan cache:clear
```

---

## After Going Live Checklist

- [ ] Domain points to correct server
- [ ] SSL certificate active
- [ ] Homepage loads without error
- [ ] Can browse and search products
- [ ] Shopping cart functional
- [ ] Checkout works
- [ ] Email sending works (test order)
- [ ] Admin login works
- [ ] Logs show no errors
- [ ] Backups running automatically
- [ ] Monitor uptime with UptimeRobot or similar

---

## Support Resources

- Laravel Docs: https://laravel.com/docs
- Laravel Deployment: https://laravel.com/docs/deployment
- cPanel Support: https://docs.cpanel.net
- Your Hosting Provider's Documentation

